/*
 * Routines for dealing with the fabric management
 *
 * The fabric topology and interconnect is described in these DB tables:
 *  enclosures - the list of enclosures and their typer
 *  linecards - the list of line card types and where they are plugged
 *  hosts - information about each host
 *  nics - information about each myrinet NIC
 *  links - links between myrinet ports (hosts or switches)
 *
 */
#include <time.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <sys/time.h> /* XXX*/

#include "db.h"
#include "libfma.h"
#include "lf_fabric.h"
#include "lf_load_fabric.h"
#include "lf_topo_map.h"
#include "lf_scheduler.h"
#include "lf_alert.h"
#include "lf_fms_comm.h"

#include "fms.h"
#include "fms_error.h"
#include "fms_fabric.h"
#include "fms_fabric_delta.h"
#include "fms_fma.h"
#include "fms_alert.h"
#include "fms_notify.h"
#include "fms_state.h"
#include "fms_fma_map.h"

static void fms_fill_enclosure_private(struct lf_enclosure *ep);
static void fms_fill_xbar_private(struct lf_xbar *xp);
static void fms_fill_lc_xcvr_private(struct lf_xcvr *xp);
static void fms_free_link(struct fms_link *linkp);
static struct lf_fabric *fms_convert_topo_to_raw_fabric(
    struct lf_topo_map *topo);
static void fms_fill_fabric_private(struct lf_fabric *fp);
static struct fms_link *fms_alloc_link(void);
#ifdef CLONE
static struct lf_fabric *fms_clone_fabric(struct lf_fabric *fp);
#endif

void
dumpit(
  struct lf_fabric *fp)
{
  int i,n;

  if (F.debug < 2) return;

  for (i=0; i<fp->num_hosts; ++i) {
   lf_host_t *hp;
   hp = fp->hosts[i];
   if (hp == NULL) { printf("%d: null\n", i); continue; }
   printf("\"%s\", %d NICs\n", hp->hostname, hp->num_nics);
   for (n=0; n<hp->num_nics;++n) {
     lf_nic_t *np;
     np = hp->nics[n];
     printf("\t%d " LF_MAC_FORMAT " sn=%s, pn=%s, fl=0x%x\n", n,
       LF_MAC_ARGS(np->mac_addr),
       np->serial_no, np->product_id, hp->fma_flags);
   }
  }

  printf("%d enclosures\n", fp->num_enclosures);
  for (i=0; i<fp->num_enclosures; ++i) {
    lf_enclosure_t *ep;
    ep = fp->enclosures[i];
    printf("  %d %s %s %d/%d slots\n",
      i, ep->name, ep->product_id, ep->num_lc_slots, ep->num_bp_slots);
    for (n=0; n<ep->num_slots; ++n) {
      lf_linecard_t *lp;
      int x;

      lp = ep->slots[n];
      if (lp == NULL) continue;
      printf("    slot %d %s (%d %d)\n",
	  lf_slot_display_no(lp), lp->product_id, lp->num_xbars, lp->num_xcvrs);

      for (x=0; x<lp->num_xbars; ++x) {
	struct lf_xbar *xp;
	int p;

	xp = LF_XBAR(lp->xbars[x]);
	printf("      xbar[%d] - xid=%d, %d ports, level=%d(%d)\n", x,
	    xp->xbar_id, xp->num_ports,
	    xp->clos_level,
	    xp->min_nic_dist);
	for (p=0; p<xp->num_ports; ++p) {
	  struct fms_link *linkp;
	  union lf_node *onp;

	  printf("       %d -", p);

	  onp = xp->topo_ports[p];
	  if (onp == NULL) {
	    printf("\n");
	    continue;
	  }

	  if (onp->ln_type == LF_NODE_XBAR) {
	    struct lf_xbar *oxp;
	    oxp = LF_XBAR(onp);
	    printf(" %s:s%d:x%d:p%d[%d]", oxp->linecard->enclosure->name,
		lf_slot_display_no(oxp->linecard), oxp->xbar_no,
		xp->topo_rports[p], oxp->clos_level);
	  } else if (onp->ln_type == LF_NODE_NIC) {
	    struct lf_nic *onicp;
	    onicp = LF_NIC(onp);
	    printf(" %s:s%d:p%d", onicp->host->hostname, onicp->host_nic_id,
		xp->topo_rports[p]);
	  } else {
	    printf(" ????");
	  }

	  printf(" %s", xp->link_state[p] == LF_LINK_STATE_UP ? "UP" : "DOWN");

	  linkp = FMS_XBAR(xp)->x_port_data[p]->link;
	  if (xp->verifiers[p].ver_nicp != NULL) {
	    struct lf_nic *nicp;
	    nicp = xp->verifiers[p].ver_nicp;

	    if (nicp == LF_VERIFY_OTHER_END) {
	      printf("<VOE>");
	    } else {
	      printf("<%s>", xp->verifiers[p].ver_nicp->host->hostname);
	    }
	  }
	  printf("ti=%d\n", xp->link_topo_index[p]);
	}
      }
    }
  }
}

int
init_fabric_vars()
{
  struct fms_fabvars *fvp;

  /* allocate space for fabric variables */
  LF_CALLOC(fvp, struct fms_fabvars, 1);
  F.fabvars = fvp;

  /* fill in default productinfo dir name */
  fvp->product_info_dir = "productinfo";

  return 0;

 except:
  if (F.fabvars != NULL) free(F.fabvars);
  return -1;
}

/*
 * Do any fabric initialization needed
 */
int
init_fabric()
{
  F.fabvars->auto_commit = ~0;	/* auto-commit everything */

  return 0;
}

/*
 * Load fabric information
 */
int
load_fabric()
{
  struct fms_fabvars *fv;
  struct lf_fabric *fp;
  struct lf_fabric_db *fdp;

  fv = F.fabvars;

  /* load up fabric definition */
  fdp = lf_fabric_db_handle(F.database);
  if (fdp == NULL) LF_ERROR(("Getting fabric db handle"));
  F.fabvars->fabric_db = fdp;

  fp = lf_load_fabric(fdp);
  if (fp == NULL) goto except;
  fv->fabric = fp;

  fms_notify(FMS_EVENT_INFO, "Fabric loaded, %d hosts", fp->num_hosts);

  /* Fill in FMS-private struct */
  fms_fill_fabric_private(fp);

#ifdef CLONE
  /* create a reference clone of this fabric */
  fv->db_fabric = fms_clone_fabric(fp);
  if (fv->db_fabric == NULL) LF_ERROR(("Error cloning fabric."));
#endif

  /* update all fabric calculations */
  fms_state_fabric_changed(FMS_STATE_FABRIC_CHANGE);

  return 0; 

 except:
  if (fv->fabric != NULL) lf_free_fabric(fv->fabric);
#ifdef CLONE
  if (fv->db_fabric != NULL) lf_free_fabric(fv->db_fabric);
#endif
  return -1;
}

/*
 * Fill in FMS private structs
 */
static void
fms_fill_fabric_private(
  struct lf_fabric *fp)
{
  int e;
  int h;
  struct fms_fabric *ffp;

  /* FMS-specific fabric variables */
  LF_CALLOC(fp->user.fms, struct fms_fabric, 1);
  ffp = FMS_FABRIC(fp);

  /* Assign FMS-specific structures to fabric elements */
  for (e=0; e<fp->num_enclosures; ++e) {
    fms_fill_enclosure_private(fp->enclosures[e]);
  }

  /* Assign FMS-specific structs for hosts */
  for (h=0; h<fp->num_hosts; ++h) {
    fms_fill_host_private(fp->hosts[h]);
  }
  return;

 except:
  fms_perror_exit(1);
}

/*
 * Fill in private linecard xcvr structs
 */
static void
fms_fill_lc_xcvr_private(
  struct lf_xcvr *xp)
{
  /* fill in user struct */
  LF_CALLOC(xp->user.fms_lc, struct fms_lc_xcvr, 1);

  /* initialize alerts */
  fms_fma_init_xcvr_alerts(xp);

  return;

 except:
  fms_perror_exit(1);
}

void
fms_free_lc_xcvr_private(
  struct lf_xcvr *xp)
{
  /* cancel all the alerts from this xcvr */
  fms_cancel_subject_alerts(FMS_LC_XCVR(xp)->alerts_anchor);
  LF_FREE(FMS_LC_XCVR(xp)->alerts_anchor);

  LF_FREE(xp->user.fms_lc);
}

/*
 * Fill in private xbar structs
 */
static void
fms_fill_xbar_private(
  struct lf_xbar *xp)
{
  int p;

  /* fill in user struct */
  LF_CALLOC(xp->user.fms, struct fms_xbar, 1);
  LF_CALLOC(FMS_XBAR(xp)->x_port_data, struct fms_xbar_port *, xp->num_ports);

  /* initialize alerts */
  fms_fma_init_xbar_alerts(xp);

  /* allocate per-port data
   * (do it all here, else link to self can bite us)
   */
  for (p=0; p<xp->num_ports; ++p) {
    struct fms_xbar_port *fxpp;

    LF_CALLOC(fxpp, struct fms_xbar_port, 1);
    FMS_XBAR(xp)->x_port_data[p] = fxpp;
  }

  /* fill in link information for each port */
  for (p=0; p<xp->num_ports; ++p) {
    union lf_node *np2;
    int p2;
    struct fms_link *linkp;

    /* get remote node and port */
    np2 = xp->topo_ports[p];
    p2 = xp->topo_rports[p];

    /* get link from remote node */
    linkp = fms_get_node_link(np2, p2);

    /* if no linkp, create one */
    if (linkp == NULL) {
      linkp = fms_alloc_link();
    }

    /* save our pointer to link struct */
    FMS_XBAR(xp)->x_port_data[p]->link = linkp;
    ++linkp->fl_ref_cnt;
  }
  return;

 except:
  fms_perror_exit(1);
}

/*
 * Allocate a link struct
 */
static struct fms_link *
fms_alloc_link()
{
  struct fms_link *linkp;

  /* allocate the link struct */
  LF_CALLOC(linkp, struct fms_link, 1);

  /* initialize alerts */
  fms_init_link_alerts(linkp);

  return linkp;

 except:
  fms_perror_exit(1);
  return NULL;
}

/*
 * Free xbar private stuff
 */
void
fms_free_xbar_private(
  struct lf_xbar *xp)
{
  int p;

  /* cancel all the alerts from this xbar */
  fms_cancel_subject_alerts(FMS_XBAR(xp)->alerts_anchor);
  LF_FREE(FMS_XBAR(xp)->alerts_anchor);

  /* dereference all the links and free the array */
  for (p=0; p<xp->num_ports; ++p) {
    fms_dereference_link(FMS_XBAR(xp)->x_port_data[p]->link);
    LF_FREE(FMS_XBAR(xp)->x_port_data[p]);
  }
  LF_FREE(FMS_XBAR(xp)->x_port_data);

  /* then free the whole user struct */
  LF_FREE(xp->user.fms);
}

/*
 * Fill in private structs for a linecard
 */
void
fms_fill_linecard_private(
  struct lf_linecard *lp)
{
  int n;

  LF_CALLOC(lp->user.fms, struct fms_linecard, 1);

  /* initialize alerts */
  fms_fma_init_linecard_alerts(lp);

  /* fill in some stuff for each xbar on the linecard */
  for (n=0; n<lp->num_xbars; ++n) {
    fms_fill_xbar_private(LF_XBAR(lp->xbars[n]));
  }

  /* fill in some stuff for each transceiver on the linecard */
  for (n=0; n<lp->num_xcvrs; ++n) {
    fms_fill_lc_xcvr_private(LF_XCVR(lp->xcvrs[n]));
  }

  FMS_LC(lp)->new = TRUE;	/* this linecard is new */

  return;

 except:
  fms_perror_exit(1);
}

/*
 * Free linecard private stuff
 */
void
fms_free_linecard_private(
  struct lf_linecard *lp)
{
  int n;

  /* free private struct for each xbar on the linecard */
  for (n=0; n<lp->num_xbars; ++n) {
    fms_free_xbar_private(LF_XBAR(lp->xbars[n]));
  }

  /* free private struct for each transceiver on the linecard */
  for (n=0; n<lp->num_xcvrs; ++n) {
    fms_free_lc_xcvr_private(LF_XCVR(lp->xcvrs[n]));
  }

  /* cancel all the alerts from this linecard */
  fms_cancel_subject_alerts(FMS_LC(lp)->alerts_anchor);
  LF_FREE(FMS_LC(lp)->alerts_anchor);

  /* then free the whole user struct */
  LF_FREE(lp->user.fms);
}

/*
 * Fill in private structs for an enclosure
 */
static void
fms_fill_enclosure_private(
  struct lf_enclosure *ep)
{
  int s;
  struct lf_linecard *lp;
  static int eid=0;

  /* assign a per-enclosure struct */
  LF_CALLOC(ep->user.fms, struct fms_enclosure, 1);
  FMS_ENC(ep)->eid = eid++;

  /* If name starts with FMS_BOGUS_SWITCH_STR, it is bogus... */
  if (strncmp(ep->name, FMS_BOGUS_SWITCH_STR, strlen(FMS_BOGUS_SWITCH_STR))
      == 0) {
    int id;

    FMS_ENC(ep)->is_bogus = TRUE;

    id = atoi(ep->name + strlen(FMS_BOGUS_SWITCH_STR));
    if (id > F.fabvars->bogus_switch_index) {
      F.fabvars->bogus_switch_index = id;
    }
  }

  /* initialize alerts */
  fms_fma_init_enclosure_alerts(ep);

  /* loop through all the line cards */
  for (s=0; s<ep->num_slots; ++s) {

    /* be careful, because not every slot has a linecard */
    lp = ep->slots[s];
    if (lp != NULL) {
      fms_fill_linecard_private(lp);
    }
  }
  return;

 except:
  fms_perror_exit(1);
}

/*
 * Free private structs for an enclosure
 */
void
fms_free_enclosure_private(
  struct lf_enclosure *ep)
{
  int s;
  struct lf_linecard *lp;

  /* cancel all alerts */
  fms_cancel_subject_alerts(FMS_ENC(ep)->alerts_anchor);
  LF_FREE(FMS_ENC(ep)->alerts_anchor);

  /* If there is a monitor running on this enclosure, shut it down */
  if (FMS_ENC(ep)->monitor != NULL) {
    LF_ERROR(("XXX Cannot kill switch monitors yet"));
  }

  /* loop through all the line cards */
  for (s=0; s<ep->num_slots; ++s) {

    /* be careful, because not every slot has a linecard */
    lp = ep->slots[s];
    if (lp != NULL) {
      fms_free_linecard_private(lp);
    }
  }

  /* free private struct */
  LF_FREE(ep->user.fms);

  return;

 except:
  fms_perror_exit(1);
}

/*
 * Clone an enclosure and appropriate private structs
 */
struct lf_enclosure *
fms_clone_enclosure(
  struct lf_enclosure *ep)
{
  struct lf_enclosure *nep;

  /* clone the enclosure and attached linecards */
  nep = lf_clone_enclosure(ep);
  if (nep == NULL) return NULL;

  /* Fill and copy enclosure private state */
  fms_fill_enclosure_private(nep);
  FMS_ENC(nep)->is_bogus = FMS_ENC(ep)->is_bogus;

  return nep;
}

/*
 * Fill in private structs for a nic
 */
void
fms_fill_nic_private(
  struct lf_nic *nicp)
{
  int p;

  LF_CALLOC(nicp->user.fms, struct fms_nic, 1);
  LF_CALLOC(FMS_NIC(nicp)->nic_port_data, struct fms_nic_port *,
	    nicp->num_ports);

  for (p=0; p<nicp->num_ports; ++p) {
    struct fms_nic_port *fnpp;
    struct fms_link *linkp;
    union lf_node *np2;
    int p2;

    /* allocate per-port struct */
    LF_CALLOC(fnpp, struct fms_nic_port, 1);
    FMS_NIC(nicp)->nic_port_data[p] = fnpp;

    /* mark this as unknown */
    lf_set_nic_link_state(nicp, p, LF_LINK_STATE_UNKNOWN);

    /* get remote node and port */
    np2 = nicp->topo_ports[p];
    p2 = nicp->topo_rports[p];

    /* get link from remote node */
    linkp = fms_get_node_link(np2, p2);

    /* if no linkp, create one */
    if (linkp == NULL) {
      linkp = fms_alloc_link();
    }

    /* save our pointer to link struct */
    fnpp->link = linkp;
    ++linkp->fl_ref_cnt;
  }

  return;

 except:
  fms_perror_exit(1);
}

/*
 * Free the private structs for a nic
 */
void
fms_free_nic_private(
  struct lf_nic *np)
{
  int p;

  /* release each link struct */
  for (p=0; p<np->num_ports; ++p) {
    fms_dereference_link(FMS_NIC(np)->nic_port_data[p]->link);
    LF_FREE(FMS_NIC(np)->nic_port_data[p]);
  }
  
  LF_FREE(FMS_NIC(np)->nic_port_data);
  LF_FREE(np->user.fms);
}

/*
 * Free a NIC which may have FMS-private data
 */
void
fms_free_nic(
  struct lf_nic *np)
{
  fms_free_nic_private(np);
  lf_free_nic(np);
}

/*
 * free the whole fabric
 */
void
fms_free_fabric(
  struct lf_fabric *fp)
{
  int e;
  int h;

  /* for now, require it to be empty */
  for (e=0; e<fp->num_enclosures; e++) {
    fms_free_enclosure_private(fp->enclosures[e]);
  }

  for (h=0; h<fp->num_hosts; h++) {
    fms_free_host_private(fp->hosts[h]);
  }

  if (fp->user.fms != NULL) {
    LF_FREE(FMS_FABRIC(fp)->topo_map);
    LF_FREE(fp->user.fms);
  }
  lf_free_fabric(fp);
}

/*
 * Fill in private structs for a host
 */
void
fms_fill_host_private(
  struct lf_host *hp)
{
  int n;

  /* assign a per-host struct */
  LF_CALLOC(hp->user.fms, struct fms_host, 1);

  /* initialize alert data */
  fms_fma_init_host_alerts(hp);

  /* loop through all the NICs */
  for (n=0; n<hp->num_nics; ++n) {
    fms_fill_nic_private(hp->nics[n]);
  }
  return;

 except:
  fms_perror_exit(1);
}

void
fms_free_host_private(
  struct lf_host *hp)
{
  int n;

  /* free all NIC-private structs */
  for (n=0; n<hp->num_nics; ++n) {
    fms_free_nic_private(hp->nics[n]);
  }

  /* cancel all alerts */
  fms_cancel_subject_alerts(FMS_HOST(hp)->alerts_anchor);
  LF_FREE(FMS_HOST(hp)->alerts_anchor);

  LF_FREE(hp->user.fms);
}


void
fms_free_host(
  struct lf_host *hp)
{
  int n;

  for (n=0; n<hp->num_nics; ++n) {
    fms_free_nic(hp->nics[n]);
  }
  fms_free_host_private(hp);
  lf_free_host(hp);
}

/*
 * Clone a host and appropriate private structs
 */
struct lf_host *
fms_clone_host(
  struct lf_host *hp)
{
  struct lf_host *nhp;

  /* clone the host and attached NICs */
  nhp = lf_clone_host(hp);
  if (nhp == NULL) return NULL;

  /* Fill and copy host private state */
  fms_fill_host_private(nhp);
  FMS_HOST(nhp)->is_bogus = FMS_HOST(hp)->is_bogus;

  return nhp;
}

/*
 * Get the link pointer for a node
 */
struct fms_link *
fms_get_node_link(
  lf_node_t *np,
  int p)
{
  if (np != NULL) {
    if (np->ln_type == LF_NODE_XBAR) {
      if (FMS_XBAR_N(np) != NULL) {
	return FMS_XBAR_N(np)->x_port_data[p]->link;
      } 
    } else if (np->ln_type == LF_NODE_NIC) {
      if (FMS_NIC_N(np) != NULL) {
	return FMS_NIC_N(np)->nic_port_data[p]->link;
      } 
    }
  }

  return NULL;
}

/*
 * Set the link pointer for a node
 */
void
fms_set_node_link(
  lf_node_t *np,
  int p,
  struct fms_link *lp)
{
  if (np != NULL) {
    if (np->ln_type == LF_NODE_XBAR) {
      FMS_XBAR_N(np)->x_port_data[p]->link = lp;
    } else if (np->ln_type == LF_NODE_NIC) {
      FMS_NIC_N(np)->nic_port_data[p]->link = lp;
    }
  }
}

/*
 * Clear the breadth-first serial number for all xbars
 */
void
fms_clear_bf_serial(
  struct lf_fabric *fp)
{
  int x;

  for (x=0; x<fp->num_xbars; ++x) {
    FMS_XBAR(fp->xbars[x])->bf_serial = 0;
  }
}

/*
 * Note than an xbar port has been oberved to be up.  If the port is
 * just "DOWN" mark it as up, but don't change it if in other states, like
 * "MAINTENANCE".
 */
void
fms_xbar_link_up(
  struct lf_xbar *xp,
  int port)
{
  union lf_node *onp;
  struct lf_host *hp;

  /* If is currently down or unknown, mark it up */
  if (xp->link_state[port] == LF_LINK_STATE_DOWN
      || xp->link_state[port] == LF_LINK_STATE_UNKNOWN) {
    fms_set_node_link_state(LF_NODE(xp), port, LF_LINK_STATE_UP);
  }

  /* If the other end is a NIC, and its owning host is disconnected, 
   * mark it NOT disconnected.
   */
  onp = xp->topo_ports[port];
  if (onp != NULL && onp->ln_type == LF_NODE_NIC) {
    hp = LF_NIC(onp)->host;
    if (FMS_HOST(hp)->disconnected) {
      FMS_HOST(hp)->disconnected = FALSE;
      fms_notify(FMS_EVENT_INFO, "%s is no longer disconnected", hp->hostname);
    }
  }
}

/*
 * Note than an xbar port has been oberved to be down
 * Returns TRUE if this is a state change.
 */
int
fms_xbar_link_down(
  struct lf_xbar *xp,
  int port)
{
  /* If link is currently up or unknown, mark it down */
  if (xp->link_state[port] == LF_LINK_STATE_UP
      || xp->link_state[port] == LF_LINK_STATE_UNKNOWN) {
    fms_set_node_link_state(LF_NODE(xp), port, LF_LINK_STATE_DOWN);
    return TRUE;
  }
  return FALSE;
}

/*
 * Set the state for a link
 */
void
fms_set_node_link_state(
  union lf_node *np,
  int port,
  int state)
{
  if (lf_get_node_link_state(np, port) != state) {

    /* update link state in the xbar */
    lf_set_node_link_state(np, port, state);

    /* update link state in the topo map */
    fms_update_topo_link_state(np, port);

    /* fabric link state changed */
    fms_state_fabric_changed(FMS_STATE_LINK_CHANGE);
  }
}

/*
 * Make all the trickle down calculations for the fabric that FMAs need.
 * This is done as late as possible before sending a map to reduce 
 * duplicated and wasted effort.
 */
void
fms_make_fabric_calcs()
{
  struct lf_fabric *fp;
  struct fms_fabric *fmsp;

  fms_notify(FMS_EVENT_DEBUG, "Performing fabric calculations");

  fp = F.fabvars->fabric;
  fmsp = FMS_FABRIC(fp);

  lf_build_xbar_array(fp);      /* this brings fp->num_xbars up-to-date */

  /* Create a topo map */
  fms_create_topo_map(fp);
}

void
fms_push_map(void *vfp)
{
  struct lf_fabric *fp;
  struct fms_fabric *fmsp;

  fp = vfp;
  fmsp = FMS_FABRIC(fp);

  /* timer popper, clear event */
  fmsp->push_map_task = NULL;

  /* make topo map, etc. */
  if (fmsp->topo_map == NULL) {
    fms_make_fabric_calcs();
  }
  
  /* If we still have no topo map, we need to request one from an FMA
   * and defer our own map pushing.
   */
  if (fmsp->topo_map == NULL) {

    /* if there is already a map request pending, just wait for it */
    if (fmsp->request_map_task == NULL) {
      fms_fma_request_map(NULL);
    }

  /* got a topo map, send it */
  } else {
    fms_send_topo_map_to_all_needy_fma(fp);
  }
}

/*
 * Is a map push pending?
 */
int
fms_map_push_pending()
{
  return (FMS_FABRIC(F.fabvars->fabric)->push_map_task != NULL);
}

/*
 * This is called whenever a change is made to the fabric topology or
 * link state changes
 */
void
fms_schedule_map_push()
{
  struct lf_fabric *fp;
  struct fms_fabric *fmsp;

  fp = F.fabvars->fabric;
  fmsp = FMS_FABRIC(fp);

  /* If map request pending, just wait */
  if (fms_map_request_pending()) {
    return;
  }

  /* If a map push is not already scheduled, schedule one now */
  if (!fms_map_push_pending()) {
    fmsp->push_map_task = lf_schedule_event(fms_push_map, fp, 5*1000);
  }
}

/*
 * Cancel a map push
 */
void
fms_cancel_map_push()
{
  struct lf_fabric *fp;
  struct fms_fabric *fmsp;

  fp = F.fabvars->fabric;
  fmsp = FMS_FABRIC(fp);

  /* Cancel schedule map push task if there is one */
  if (fmsp->push_map_task != NULL) {
    lf_remove_event(fmsp->push_map_task);
    fmsp->push_map_task = NULL;
  }
}

/*
 * reduce reference count on a link and free the memory
 * if it goes to zero
 */
void
fms_dereference_link(
  struct fms_link *linkp)
{
  if (linkp != NULL && --linkp->fl_ref_cnt <= 0) {
    fms_free_link(linkp);
  }
}

/*
 * Free an FMS link structure and its contents
 */
static void
fms_free_link(
  struct fms_link *linkp)
{

  /* cancel all alerts */
  fms_cancel_subject_alerts(linkp->fl_alerts_anchor);
  LF_FREE(linkp->fl_alerts_anchor);

  LF_FREE(linkp);
}

/*
 * Make a fabric given a topo map we got from an FMA.  The topo map we get
 * knows nothing about host associations for NICs, so we first create a
 * whole fabric with each NIC residing in one bogus host. 
 * Then, update our current fabric with the reference fabric from the map.
 */
void
fms_process_topo_map(
  struct lf_topo_map *topo)
{
  struct lf_fabric *fp;
  struct lf_fabric *nfp;
  int rc;


  /* get the current fabric pointer */
  fp = F.fabvars->fabric;

  /* this just converts the topo map into a fabric full of bogus hosts */
  nfp = fms_convert_topo_to_raw_fabric(topo);
  if (nfp == NULL) {
    LF_ERROR(("Error making fabric from topo map"));
  }

  fms_notify(FMS_EVENT_INFO, "Map received, %d hosts, %d xbars",
      nfp->num_hosts, nfp->num_xbars);

  if (F.debug) {
    printf("======= mapping results ==========\n");
    if (F.debug) dumpit(nfp);
  }

  /* Update the current fabric from mapped fabric */
  rc = fms_update_current_fabric_with_mapped(fp, nfp);
  if (rc == -1) {
    LF_ERROR(("Unable to update current fabric from mapped fabric"));
  }
  if (0 && F.debug) {
    printf("======= primary fabric ==========\n");
    dumpit(fp);
  }

  /* free the fabric we got from the mapping */
  fms_free_fabric(nfp);

  /* signal state machine that we have a new fabric and map */
  fms_state_fabric_changed(FMS_STATE_FABRIC_CHANGE);

  return;

 except:
  fms_perror_exit(1);
}

#if 0
obsolete?
static void fms_merge_fabric_nics(struct lf_fabric *ofp, struct lf_fabric *nfp);
/*
 * merge all NICs from ofp into nfp, removing them from nfp.
 * Loop through all the hosts in ofp, moving the NICs from each host
 * to nfp.  The link information from nfp supercedes that in ofp, so 
 * all we are really doing is migrating the host structs from ofp to nfp.
 * If any NICs are in ofp but not nfp, remove all their link state and copy
 * them into nfp.
 */
static void
fms_merge_fabric_nics(
  struct lf_fabric *ofp,
  struct lf_fabric *nfp)
{
  int h;
  int rc;

  /* Process each host in the old fabric */
  for (h=0; h<ofp->num_hosts; ++h) {
    struct lf_host *hp;
    struct lf_host *nhp;
    int n;

    hp = ofp->hosts[h];
    nhp = fms_dup_host(hp);

    /* first, copy all relevant NIC from the new fabric to this host */
    for (n=0; n<hp->num_nics; ++n) {
      struct lf_nic *onicp;
      struct lf_nic *nnicp;

      /* see if this NIC is in the new fabric */
      onicp = hp->nics[n];
      nnicp = lf_find_nic_by_mac(nfp, onicp->mac_addr);

      /* If present in new fabric, make sure the host is bogus.
       * Then, copy over anything relevant from
       * the old one, and install the new one in this host.
       *
       * If not present, unlink any connections and leave in place.
       */
      if (nnicp != NULL) {
	struct lf_host *nhp;

	/* the owner of this NIC must (for now) be a bogus host */
	nhp = nnicp->host;
	if (!FMS_HOST(nhp)->is_bogus) {
	  LF_ERROR(("Nic host in new fabric not bogus!"));
	}

	/* remove this NIC from the host, and remove the host
	 * if no more NICs
	 */
	lf_remove_nic_from_host(nnicp);
	if (nhp->num_nics <= 0) {
	  lf_remove_host_from_fabric(nfp, nhp);
	  fms_free_host(nhp);
	}

	/* copy serial number and product ID from old one */
	LF_DUP_STRING(nnicp->serial_no, onicp->serial_no);
	LF_DUP_STRING(nnicp->product_id, onicp->product_id);
	nnicp->slot = n;
	nnicp->host_nic_id = onicp->host_nic_id;
	nnicp->host = hp;

	/* aqttach this NIC to the old host */
	hp->nics[n] = nnicp;

	fms_free_nic(onicp);	/* free the old NIC */

      /* NIC not present in new fabric, so just create a new
       * one and add it to the new host
       */
      } else {
	nnicp = fms_dup_nic(onicp);
      }
    }

    /* remove this host from old fabric and add to the new */
    /* XXX  lf_remove_host_from_fabric(ofp, hp); */
    rc = lf_add_existing_host_to_fabric(nfp, nhp);
    if (rc == -1) LF_ERROR(("Error adding host to fabric"));
  }
  return;

 except:
  fms_perror_exit(1);
}
#endif

/*
 * Generate a fabric based on a topo map we got back from an FMA
 */
static struct lf_fabric *
fms_convert_topo_to_raw_fabric(
  struct lf_topo_map *topo)
{
  struct lf_fabric *fp;

  /* allocate fabric struct (FMS-private is dealt with elsewhere) */
  LF_CALLOC(fp, struct lf_fabric, 1);

  /* Add all the NICs, creating a bogus host for each one */
  {
    struct lf_topo_nic *tnp;
    int num_nics;
    int n;

    /* number of NICs will also be number of bogus hosts */
    num_nics = ntohl(topo->num_nics_32);

    /* allocate hosts array */
    LF_CALLOC(fp->hosts, struct lf_host *, num_nics);
    fp->num_hosts = num_nics;

    /* loop through each NIC */
    tnp = LF_TOPO_NIC_ARRAY(topo);
    for (n=0; n<num_nics; ++n, ++tnp) {
      struct lf_host *hp;
      struct lf_nic *nicp;
      lf_string_t hostname;

      /* allocate and fill in what we can of the host */
      LF_CALLOC(hp, struct lf_host, 1);
      LF_CALLOC(hp->nics, struct lf_nic *, 1);
      fp->hosts[n] = hp;

      /* give it a bogus hostname */
      lf_make_mac_hostname(hostname, tnp->mac_addr, tnp->num_ports_8);
      LF_DUP_STRING(hp->hostname, hostname);

      /* allocate the NIC based on number of ports */
      nicp = lf_alloc_generic_nic(tnp->num_ports_8);
      if (nicp == NULL) LF_ERROR(("Error allocating NIC"));

      /* Fill in NIC fields */
      LF_MAC_COPY(nicp->mac_addr, tnp->mac_addr);
      nicp->slot = 0;
      nicp->host_nic_id = 0;
      nicp->host = hp;
      nicp->mag_id = ntohs(tnp->mag_id_16);

      /* XXX - we can actually group the NICs into hosts, since
       * the topo map carries that information
       */

      /* attach this NIC to the host */
      hp->nics[0] = nicp;
      hp->num_nics = 1;
      hp->fw_type = tnp->firmware_type_8;
      hp->fma_flags = ntohs(tnp->flags_16);

      fms_fill_host_private(hp);	/* fill in FMS-private fields */
      FMS_HOST(hp)->is_bogus = TRUE;	/* this is not a real host */
    }
  }

  /* Now create a bunch of bogus enclosures for the xbars in the topo map.
   * Each will be a bogus GENSW switch with a GENLC*P linecard.
   */
  {
    struct lf_topo_xbar *txp;
    int num_xbars;
    int x;

    /* number of xbars will also be number of enclosures */
    num_xbars = ntohl(topo->num_xbars_32);

    /* allocate enclosures array */
    LF_CALLOC(fp->enclosures, struct lf_enclosure *, num_xbars);
    fp->num_enclosures = num_xbars;

    /* loop through all the xbars */
    txp = LF_TOPO_XBAR_ARRAY(topo);
    for (x=0; x<num_xbars; ++x, ++txp) {
      struct lf_enclosure *ep;
      lf_string_t enc_name;
      struct lf_linecard *lp;
      struct lf_xbar *xp;
      int xbar_id;
      lf_string_t serial_no;

      /* allocate the enclosure */
      ++F.fabvars->bogus_switch_index;
      sprintf(enc_name, FMS_BOGUS_SWITCH_STR "%d",
	  F.fabvars->bogus_switch_index);
      ep = lf_allocate_enclosure("GENSW", enc_name);
      if (ep == NULL) LF_ERROR(("Error allocating enclosure"));
      fp->enclosures[x] = ep;

      /* get xid and serial number (same for bogus linecards) */
      xbar_id = ntohl(txp->xbar_id_32);
      sprintf(serial_no, "%d", xbar_id);

      /* allocate linecard to hold the xbar */
      lp = lf_allocate_generic_linecard(ep, 0, ntohs(txp->num_ports_16),
	                                serial_no);
      if (lp == NULL) LF_ERROR(("Error allocating linecard"));

      fms_fill_enclosure_private(ep);	/* Fill in FMS-private fields */
      FMS_ENC(ep)->is_bogus = TRUE;	/* this is not a real enclosure */

      /* Fill in xbar ID */
      xp = LF_XBAR(lp->xbars[0]);
      xp->xbar_id = xbar_id;
      xp->quadrant_disable = txp->quadrant_disable_8;
    }
  }

  /* finally, make all the links */
  {
    struct lf_topo_link *tlp;
    int num_links;
    int l;

    /* loop through all the links */
    num_links = ntohl(topo->num_links_32);
    tlp = LF_TOPO_LINK_ARRAY(topo);
    for (l=0; l<num_links; ++l, ++tlp) {
      union lf_node *np[2];
      int port[2];
      int node_id;
      int index;
      int i;

      /* We get to take a shortcut here, since we know that the topo
       * map indices are the same as the fabric indices (for the moment)
       */
      for (i=0; i<2; ++i) {
	node_id = ntohl(tlp->node_index_32[i]);
	port[i] = LF_TOPO_NI_PORT(node_id);
	index = LF_TOPO_NI_INDEX(node_id);
	if (LF_TOPO_NI_XBAR(node_id)) {
	  np[i] = fp->enclosures[index]->slots[0]->xbars[0];
	} else {
	  np[i] = LF_NODE(fp->hosts[index]->nics[0]);
	}
      }

      /* link the two ports */
      fms_connect_topo_nodes(np[0], port[0], np[1], port[1]);
    }
  }

  /* start with a good xbar array */
  lf_build_xbar_array(fp);

  return fp;

 except:
  fms_perror_exit(1);
  return NULL;
}

/*
 * Make a link between two topo nodes -
 * make the topo link, and also make the physical links that go with it.
 * Since both sides may already have a linkp struct, deference one if needed.
 * We assume here that both ends are already unlinked and do not bother with
 * unlinking them.
 */
void
fms_connect_topo_nodes(
  union lf_node *np1,
  int port1,
  union lf_node *np2,
  int port2)
{
  struct fms_link *lp1;
  struct fms_link *lp2;

  /* there can be only one...  remove lp2 if necessary */
  lp1 = fms_get_node_link(np1, port1);
  lp2 = fms_get_node_link(np2, port2);
  if (lp1 == NULL) {
    if (lp2 == NULL) {
      lp1 = fms_alloc_link();
      if (lp1 == NULL) LF_ERROR(("Error allocating link"));
    } else {
      lp1 = lp2;
    }
  } else if (lp2 != NULL && lp1 != lp2) {
    fms_dereference_link(lp2);
  }

  /* make lp1 the link struct for both nodes */
  fms_set_node_link(np1, port1, lp1);
  fms_set_node_link(np2, port2, lp1);

  /* no-connect and self-connect get ref_count of 1 */
  if (np2 == NULL || (np1 == np2 && port1 == port2)) {
    lp1->fl_ref_cnt = 1;
  } else {
    lp1->fl_ref_cnt = 2;
  }

  /* link the nodes */
  lf_connect_topo_nodes(np1, port1, np2, port2);

  /* set the link state to up */
  _lf_set_node_link_state(np1, port1, LF_LINK_STATE_UP);
  _lf_set_node_link_state(np2, port2, LF_LINK_STATE_UP);

  return;

 except:
  fms_perror_exit(1);
}

/*
 * Adjust all the port numbers on an xbar by "offset".
 * We not only need to adjust our pointers, but we need to adjust the
 * rport of the other end.
 */
int
fms_adjust_xbar_port_nos(
  struct lf_xbar *xp,
  int offset)
{
  int p;
  int ap;		/* adjusted port number */
  int incr;

  /* If positive offset, start at the top and work down, else work up */
  if (offset > 0) {
    p = xp->num_ports-1;
    incr = -1;
  } else if (offset < 0) {
    p = 0;
    incr = 1;
  } else {
    return 0;	/* If nothing to do, just return! */
  }

  /* This loop goes backwards or forwards, depending on offset */
  do {
    union lf_node *onp;
    struct lf_xcvr *xcp;

    ap = p + offset;

    /* adjust topo port first */
    onp = xp->topo_ports[p];
    if (onp != NULL) {

      if (ap >= xp->num_ports || ap < 0) {
	LF_ERROR(("Error: topo port adjustment would go out of bounds"));
      }
      if (xp->topo_ports[ap] != NULL) {
	LF_ERROR(("Adjustment target topo port not empty!"));
      }

      /* make the new links */
      lf_make_topo_link(onp, xp->topo_rports[p], LF_NODE(xp), ap);
      lf_make_topo_link(LF_NODE(xp), ap, onp, xp->topo_rports[p]);
      xp->topo_ports[p] = NULL;

      /* Move any link structs */
      if (FMS_XBAR(xp)->x_port_data[ap]->link != NULL) {
	fms_dereference_link(FMS_XBAR(xp)->x_port_data[ap]->link);
	FMS_XBAR(xp)->x_port_data[ap]->link = FMS_XBAR(xp)->x_port_data[p]->link;
	FMS_XBAR(xp)->x_port_data[p]->link = NULL;
      }
    }

    /* adjust physical port */
    xcp = LF_XCVR(xp->phys_ports[p]);
    if (xcp != NULL) {
      int xcport;

      xcport = xp->phys_rports[p] - xcp->num_conns;
      onp = xcp->ports[xcport];

      if (onp != NULL) {
	struct lf_xcvr *axcp;
	int axcport;

	if (ap >= xp->num_ports || ap < 0) {
	  LF_ERROR(("Error: phys port adjustment would go out of bounds"));
	}
	axcp = LF_XCVR(xp->phys_ports[ap]);
	if (axcp == NULL) {
	  LF_ERROR(("Missing xcvr!"));
	}
	axcport = xp->phys_rports[ap] - xcp->num_conns;
	if (axcp->ports[axcport] != NULL) {
	  LF_ERROR(("Adjustment target phys port not empty!"));
	}

	/* make the new links */
	lf_make_phys_link(onp, xcp->rports[xcport], LF_NODE(axcp), axcport);
	lf_make_phys_link(LF_NODE(axcp), axcport, onp, xcp->rports[xcport]);
	xcp->ports[xcport] = NULL;
      }
    }

    p += incr;
  } while (p >= 0 && p < xp->num_ports);

  return 0;

 except:
  return -1;
}

/*
 * Remove an enclosure from primary fabric, DB fabric, and DB
 */
void
fms_remove_enclosure(
  struct lf_enclosure *ep)
{
  struct lf_fabric *fp;
  
  /* remove from database */
  fms_remove_enclosure_from_db(ep);
  
  /* remove this from primary fabric */
  fp = F.fabvars->fabric;
  fms_remove_enclosure_from_fabric(fp, ep);
}

/*
 * Remove an enclosure from a fabric
 */
void
fms_remove_enclosure_from_fabric(
  struct lf_fabric *fp,
  struct lf_enclosure *ep)
{
  /* remove this from the fabric */
  lf_remove_enclosure_from_fabric(fp, ep);

  /* free FMS-specific structs of enclosure and children */
  fms_free_enclosure_private(ep);

  /* free the libfma structs */
  lf_free_enclosure(ep);
}

#ifdef CLONE
/*
 * Clone the fabric and save some notes
 */
static struct lf_fabric *
fms_clone_fabric(
  struct lf_fabric *fp)
{
  struct lf_fabric *cfp;
  int e;
  int h;

  /* clone the fabric */
  cfp = lf_clone_fabric(fp);
  if (cfp == NULL) LF_ERROR(("Error cloning fabric"));

  /* Find clone equivalent for each enclosure */
  for (e=0; e<fp->num_enclosures; ++e) {
    struct lf_enclosure *ep;
    struct lf_enclosure *cep;

    ep = fp->enclosures[e];
    cep = lf_find_enclosure_by_name(cfp, ep->name);

    FMS_ENC(ep)->clone_mate = cep;
  }

  /* Find clone equivalent for each host */
  for (h=0; h<fp->num_hosts; ++h) {
    struct lf_host *hp;
    struct lf_host *chp;

    hp = fp->hosts[h];
    chp = lf_find_host_by_name(cfp, hp->hostname);

    FMS_HOST(hp)->clone_mate = chp;
  }

  return cfp;

 except:
  return NULL;
}
#endif

/*
 * Allocate a new linecard
 */
struct lf_linecard *
fms_new_linecard(
  struct lf_enclosure *ep,
  int slot,
  char *product_id,
  char *serial_no)
{
  struct lf_linecard *lp;

  lp = lf_allocate_linecard(ep, slot, product_id, serial_no);
  if (lp == NULL) {
    LF_ERROR(("Error allocating new linecard"));
  }

  /* Add to the database and database fabric */
  fms_add_linecard_to_db(lp);

  /* Fill in FMS-private info */
  fms_fill_linecard_private(lp);
  return lp;

 except:
  fms_perror_exit(1);
  return NULL;
}

/*
 * Report whether we think DB is complete or not.  
 * Scan through the list of switches and return FALSE if any
 * are bogus.
 */
int
fms_db_complete()
{
  struct lf_fabric *fp;
  struct lf_enclosure *ep;
  int e;

  fp = F.fabvars->fabric;
  for (e=0; e<fp->num_enclosures; ++e) {
    ep = fp->enclosures[e];
    if (FMS_ENC(ep)->is_bogus) {
      return FALSE;
    }
  }
  return TRUE;
}

/*
 * Mark all links on a linecard as being in or out of maintenance mode
 */
int
fms_linecard_maintence(
  struct lf_linecard *lp,
  int port_label,
  int up)
{
  enum lf_link_state state;

  state = up ? LF_LINK_STATE_UP : LF_LINK_STATE_MAINTENANCE;
  
  /* all ports? */
  if (port_label == -1) {
    int x;
    int xc;

    /*
     * First, do all ports on all xbars
     */
    for (x=0; x<lp->num_xbars; ++x) {
      struct lf_xbar *xp;
      int p;

      xp = LF_XBAR(lp->xbars[x]);
      for (p=0; p<xp->num_ports; ++p) {
	/* update link state in fabric and topo map */
	lf_set_node_link_state(LF_NODE(xp), p, state);
	fms_update_topo_link_state(LF_NODE(xp), p);
      }
    }

    /*
     * next, do all the transceivers.  this may duplicate work done above, 
     * but it doesn't matter
     */
    for (xc=0; xc<lp->num_xcvrs; ++xc) {
      struct lf_xcvr *xcp;
      int p;

      xcp = LF_XCVR(lp->xcvrs[xc]);

      /* just look at the internal connections */
      for (p=xcp->num_conns; p<xcp->num_ports; ++p) {
	struct lf_xbar *xp;

	xp = LF_XBAR(xcp->ports[p]);
	if (xp != NULL) {
	  int xport;

	  xport = xcp->rports[p];

	  /* update link state in fabric and topo map */
	  lf_set_xbar_link_state(xp, xport, state);
	  fms_update_topo_link_state(LF_NODE(xp), xport);
	}
      }
    }

  /*
   * If just one port specified, find the matching xcvr
   */
  } else {
    int xc;
    int p;
    struct lf_xcvr *xcp;

    /* translate port label into port number */
    for (xc=0; xc<lp->num_xcvrs; ++xc) {
      if (lp->xcvr_labels[xc] == port_label) break;
    }
    if (xc >= lp->num_xcvrs) {
      LF_ERROR(("No such port number (%d) on slot %d of %s",
	  port_label, lf_slot_display_no(lp), lp->enclosure->name));
    }

    xcp = LF_XCVR(lp->xcvrs[xc]);

    /* just look at the internal connections */
    for (p=xcp->num_conns; p<xcp->num_ports; ++p) {
      struct lf_xbar *xp;

      xp = LF_XBAR(xcp->ports[p]);
      if (xp != NULL) {
	int xport;

	xport = xcp->rports[p];

	/* update link state in fabric and topo map */
	lf_set_xbar_link_state(xp, xport, state);
	fms_update_topo_link_state(LF_NODE(xp), xport);
      }
    }
  }

  /* a little heavy handed, but necessary */
  fms_state_fabric_changed(FMS_STATE_FABRIC_CHANGE);

  return 0;

 except:
  return -1;
}

/*
 * Return TRUE/FALSE indicating whether a given node is part of a bogus
 * construct
 */
int
fms_is_bogus(
  union lf_node *np)
{
  switch (np->ln_type) {
  case LF_NODE_NIC:
    {
      struct lf_host *hp;

      /* don't do anything if other end is bogus */
      hp = LF_NIC(np)->host;
      if (hp == NULL || FMS_HOST(hp)->is_bogus) {
	return TRUE;
      }
    }
    break;
  case LF_NODE_XBAR:
    {
      struct lf_linecard *lp;
      struct lf_enclosure *ep;

      lp = LF_XBAR(np)->linecard;
      if (lp == NULL) {
	return TRUE;
      }
      ep = lp->enclosure;
      if (ep == NULL || FMS_ENC(ep)->is_bogus) {
	return TRUE;
      }
    }
    break;
  case LF_NODE_NIC_XCVR:
  case LF_NODE_LC_XCVR:
    /* XXX */
    break;
  }

  return FALSE;
}
